/* Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCDISCOVERERSUPPORTAPI_H__
#define __VNCDISCOVERERSUPPORTAPI_H__

/**
 * \file vncdiscoverersupportapi.h
 *
 * This file defines the Discoverer Support API. The Discoverers use this API
 * for communicating back to the Application, for Logging and for allocating,
 * or freeing data structures in the memory.
 *
 * This file should not be included directly. vncdiscoverer.h should be used
 * instead.
 *
 * \see VNCDiscovererSupportAPI
 * \see vncdiscoverer.h
 */

#include <vncdiscoverysdktypes.h>
#include <vncdeviceprovider.h>
#include <stddef.h>
#include <vncdiscoverysdkutils.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief The typedef for the VNCDiscoverer. The implementation should be a
 * structure with the name VNCDiscovererImpl.
 */
typedef struct VNCDiscovererImpl VNCDiscoverer;

/**
 * \brief The typedef for the VNCDiscovererEntity. The implementation of the
 * Entity can be anything and this decision is left down to each particular 
 * Discoverer implementation.
 */
typedef void VNCDiscovererEntity;

/**
 * \brief The typedef for the Device Provider structure. The implementation of
 * this is left to the Discovery SDK.
 */
typedef struct VNCDiscovererDeviceProvider VNCDiscovererDeviceProvider;

/**
 * \brief Typedef for the Discoverer dynamic context. The implementation of the
 * Entity can be anything and this decision is left down to each particular 
 * Discoverer implementation.
 */
typedef void* VNCDiscovererDynamicContext;

/**
 * \brief Notification once the Start operation is complete.
 *
 * Informs the client Application whether the start operation on the 
 * Discoverer instance was successful or not.
 *
 * The Discoverer has to notify that it is started before making any 
 * other notifications, or sending any other responses.
 *
 * If you want to set any timers you can do so after this call returns. If you
 * are restarting the Discoverer, make sure to restart any timers that you need.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that started.
 * \param error VNCDiscoverySDKErrorNone if the start operation was succesful,
 *              or otherwise another error code. If the error code is different
 *              from VNCDiscoverySDKErrorNone then the SDK will not monitor the
 *              Discoverer handles.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererNotifyStarted(void *pContext, 
                                VNCDiscoverer *pDiscoverer, 
                                VNCDiscoverySDKError error);

/**
 * \brief Notification once the Stop operation is complete.
 *
 * Informs the client application when the discoverer has stopped, either
 * unexpectedly or by a stop operation.
 *
 * Until this function is called, the Discoverer should not change the Event
 * Handler value. After this call, the SDK will not monitor the Event Handler
 * anymore and the Discoverer can modify it.
 *
 * After this call the SDK will not monitor the Event Handles of any of the
 * Entities. Also it will cancel all the timers.
 *
 * The Discoverer should not make any notification calls, or responses, after
 * it notified it's stopped.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that stopped.
 * \param error The reason that the discoverer stopped. This should be
 *              VNCDiscoverySDKErrorNone when caused by a stop operation.
 * \param error VNCDiscoverySDKErrorNone if the stop operation was succesful,
 *              or otherwise another error code.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererNotifyStopped(void *pContext, 
                                VNCDiscoverer *pDiscoverer,
                                VNCDiscoverySDKError error);

/**
 * \brief Asks the SDK to set a timer for the Discoverer.
 *
 * This call asks the SDK to call back the Discoverer after a certain time
 * period.
 *
 * The SDK will only call back once, so for further notifications this methods
 * should be called again.
 *
 * The method should only be called when the Discoverer is started, otherwise
 * it will have no effect. Make sure the SDK is notified that the Discoverer is
 * started before making this call.
 *
 * To stop an existing timer, set the timeout value to VNCDiscoverySDKNoTimeout.
 *
 * To get an instant callback, set the timeout value to VNCDisocverySDKInstantCall.
 *
 * This method must not be used if the Discoverer is using multiple
 * handles. The Discoverer must call VNCDiscovererSetTimerForHandle in that
 * case, because calling this function will have no effect.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer The Discoverer for which to set the timer.
 * \param timeout The timeout value. It should not be negative. If you want to
 *            stop the timer set to VNCDiscoverySDKNoTimeout.
 *
 * \see VNCDiscovererHandleTimeout
 * \see VNCDiscovererGetEventHandle
 * \see VNCDiscovererGetMultipleEventHandlesEx, VNCDiscovererSetTimerForHandle
 */
typedef void VNCCALL
VNCDiscovererSetTimer(void *pContext,
                                VNCDiscoverer *pDiscoverer,
                                VNCDiscoverySDKTimeoutMicroseconds timeout);

/**
 * \brief Notification that an entity has been discovered.
 *
 * Informs the client Application that an entity has been discovered.
 *
 * This method should only be called when the Discoverer is running (has been
 * started successfully).
 *
 * This method can be called for an entity that was being scanned (reported by
 * VNCDiscovererNotifyScanningEntity).
 *
 * \note The existence of an entity does not need to be definite, even if  
 * the Discoverer successfully provides a vnccmd. It does not need to guarantee 
 * that a VNC Server is present and listening for incoming connections. False 
 * positives may be reported by some Discoverers. It is ideal, but not
 * necessary, for Discoverers to ensure the VNC Server is ready to receive
 * incoming connections.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that discovered the entity.
 * \param pEntity Pointer to the newly discovered entity. Can be the same
 *                  entity as reported by VNCDiscovererNotifyScanningEntity.
 *
 * \see VNCDiscovererNotifyScanningEntity, VNCDiscovererNotifyEntityDisappeared
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityAppeared(void *pContext, 
                          VNCDiscoverer *pDiscoverer,
                          VNCDiscovererEntity *pEntity);

/**
 * \brief Asks the SDK to set a timer for an Entity.
 *
 * This call asks the SDK to call back the Entity after a certain time
 * period.
 *
 * The SDK will only call back once, so for further notifications this methods
 * should be called again.
 *
 * The method should only be called when the Discoverer is started, otherwise
 * it will have no effect.
 *
 * To stop an existing timer, set the timeout value to VNCDiscoverySDKNoTimeout.
 *
 * To get an instant callback, set the timeout value to VNCDisocverySDKInstantCall.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pEntity The Entity for which to set the timer.
 * \param timeout The timeout value.
 */
typedef void VNCCALL
VNCDiscovererSetEntityTimer(void *pContext,
                              VNCDiscovererEntity *pEntity,
                              VNCDiscoverySDKTimeoutMicroseconds timeout);

/**
 * \brief Notification that an existing entity has changed.
 *
 * Informs the client Application that an entity has changed.
 *
 * A change in the entity can mean different things for different Discoverers,
 * so a Discoverer should document what the expected changes are.
 *
 * This method should only be called when the Discoverer is running (has been
 * started successfully). Not all Discoverers need to monitor changes to the
 * entities and notify them, but some Discoverers might wish to do so.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pEntity Pointer to the changed entity.
 * \param pChangeDescription A string that describes what has changed with the
 * entity. Can be NULL. Ownership of the string stays with the Discoverer.
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityChanged(void *pContext,
    VNCDiscovererEntity *pEntity,
    const char* pChangeDescription);

/**
 * \brief Notification that an existing entity has disappeared.
 *
 * Informs the client Application that an existing entity has disappeared
 * (disconnected).
 *
 * This method should only be called when the Discoverer is running (has been
 * started successfully). Discoverers need to make sure to inform the Application
 * if entities disappear. From the time this call is made, the SDK will call 
 * destroy entity when the entity is not needed anymore. The Discoverer should
 * not destroy the entity before that time.
 *
 * This method should be called for an entity that was being scanned (reported
 * through VNCDiscovererNotifyScanningEntity), or an entity that has been
 * confirmed through VNCDiscovererNotifyEntityAppeared.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pEntity Pointer to the entity that disappeared.
 *
 * \see VNCDiscovererNotifyScanningEntity, VNCDiscovererNotifyEntityAppeared
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityDisappeared(void *pContext, 
                                VNCDiscovererEntity *pEntity);

/**
 * \brief Returns the entity value requested.
 *
 * Returns to the client Application the value for the requested key of the
 * entity. This method is in response to VNCDiscovererFetchEntityValue.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pEntity The pointer to the entity that has the value. This
 *                    needs to match what was passed in
 *                    VNCDiscovererFetchEntityValue.
 * \param pKey The string that represents the requested key. The SDK owns this
 *             string so it should be allocated using VNCDiscovererAllocString.
 *             Alternatively the original string for the key (received in
 *             VNCDiscovererFetchEntityValue) can be passed here. This
 *             needs to match what was passed in VNCDiscovererFetchEntityValue.
 *             The ownership of the string is passed to the SDK.
 * \param requestId The request ID that was passed on the original fetch call.
 *                  If the Discoverer does not support remembering requestIDs,
 *                  then this should be set to VNCDiscoverSDKNoRequestId.
 * \param error VNCDiscoverySDKErrorNone if the fetch operation was succesful,
 *              VNCDiscoverySDKErrorNotFound if the entity doesn't exist,
 *              VNCDiscoverySDKErrorOutOfMemory if there is not enough memory
 *              to complete the request, VNCDiscoverySDKErrorNotSupported if
 *              the key is not supported by the entity, or otherwise another 
 *              error code documented by the Discoverer.
 * \param pValue The string containing the response value. This string is owned
 *               by the SDK so it should be allocated using
 *               VNCDiscovererAllocString. The ownership of the string is 
 *               passed to the SDK.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererReturnEntityValue(void *pContext,
                                VNCDiscovererEntity *pEntity,
                                char *pKey,
                                VNCDiscoverySDKRequestId requestId,
                                VNCDiscoverySDKError error,
                                char *pResultValue);

/**
 * \brief Notifies of the outcome of the post operation.
 *
 * Notifies the client Application with the outcome of posting to the 
 * requested key of the entity. This method is in response to VNCDiscovererPostEntityValue.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that owns the entity. This
 *                    needs to match what was passed in
 *                    VNCDiscovererPostEntityValue.
 * \param pEntity The pointer to the entity that has the value. This
 *                    needs to match what was passed in
 *                    VNCDiscovererPostEntityValue.
 * \param pKey The string that represents the posted key. The SDK owns this
 *             string so it should be allocated using VNCDiscovererAllocString.
 *             Alternatively the original string for the key (received in
 *             VNCDiscovererPostEntityValue) can be passed here. This
 *             needs to match what was passed in VNCDiscovererPostEntityValue.
 *             The ownership of the string is passed to the SDK.
 * \param requestId The request ID that was passed on the original post call.
 *                  If the Discoverer does not support remembering requestIDs,
 *                  then this should be set to VNCDiscoverSDKNoRequestId.
 * \param error VNCDiscoverySDKErrorNone if the post operation was succesful,
 *              VNCDiscoverySDKErrorNotSupported if the key is not supported,
 *              or read-only, VNCDiscoverySDKErrorOutOfMemory if there is not
 *              enough memory to complete the request,
 *              VNCDiscoverySDKErrorInvalidParameter if the value passed to the
 *              Discoverer is NULL and the Discoverer doesn't support NULL,
 *              VNCDiscoverySDKErrorNotFound if the Entity can't be found, or 
 *              otherwise another error code.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityValuePosted(void *pContext, 
                                VNCDiscovererEntity *pEntity,
                                char *pKey,
                                char *pValue,
                                VNCDiscoverySDKRequestId requestId,
                                VNCDiscoverySDKError error);

/**
 * \brief Notifies of the outcome of the publish entity operation.
 *
 * Notifies the client Application with the outcome of publishing and entity.
 * This method is in response to VNCDiscovererPublishEntity.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that owns the entity.
 * \param pEntity The pointer to the entity that was published.
 * \param error VNCDiscoverySDKErrorNone if the publish operation was 
 *              succesful, or otherwise another error code.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityPublished(void *pContext,
                                VNCDiscovererEntity *pEntity,
                                VNCDiscoverySDKError error);

/**
 * \brief Notifies of the outcome of the unpublish entity operation.
 *
 * Notifies the client Application with the outcome of unpublishing and entity.
 * This method is in response to VNCDiscovererUnpublishEntity.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that owns the entity.
 * \param pEntity The pointer to the entity that was unpublished.
 * \param error VNCDiscoverySDKErrorNone if the unpublish operation was 
 *              succesful, or otherwise another error code.
 *
 * \see VNCDiscovererInterface, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDiscovererNotifyEntityUnpublished(void *pContext,
                                VNCDiscovererEntity *pEntity,
                                VNCDiscoverySDKError error);

/**
 * \brief Logs a string for the Discoverer.
 *
 * This method is used by the Discoverer to log to the SDK logs.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param severity The severity of the log.
 * \param pText The string to log. This is owned by the Discoverer so it should
 *              be freed after the log method returns.
 */
typedef void VNCCALL
VNCDiscovererLog(void *pContext, VNCDiscovererLogSeverity severity, 
                                const char *pText);

/**
 * \brief Allocates the memory for a string for the Discoverer.
 *
 * This method should be used if the Discoverer has to create a string and pass
 * it to the SDK. The method can allocate a new string that is a copy of the
 * original string, or can just allocate a certain amount of memory.
 * 
 * \param pString The original string. If the new string should be a copy of an
 *                original one this is where the original is specified. If the 
 *                new string is empty then this should be NULL
 * \param length The length of the string. If the first parameter is NULL then
 *               the length should be greater than 0. Otherwise, if there is an
 *               original string, the length specifies how much of the string
 *               to copy. Use -1 if you want the entire string copied.
 *
 * \return A pointer to the new string, or NULL if there is not enough memory
 *        (or NULL was passed initially together with -1 to allocString).
 */
typedef char *VNCCALL
VNCDiscovererAllocString(const char *pString, size_t length);

/**
 * \brief Frees the memory of a string allocated by the SDK.
 *
 * This is used to free any string that was allocated through the SDK.
 *
 * The provided string must be NUL terminated if non-NULL.
 *
 * \param pString The string to be freed.
 */
typedef void VNCCALL
VNCDiscovererFreeString(char *pString);

/**
 * \brief Frees the memory of an offer parameter object allocated by the SDK.
 *
 * This is used to free any offer parameter object allocated by the SDK.
 *
 * \param pValue The object to free.
 *
 * \see VNCDeviceProviderAllocOfferParameter
 */
typedef void VNCCALL
VNCDeviceProviderFreeOfferParameter(const VNCDiscovererOfferParameter *pValue);

/**
 * \brief Allocates a new key-value pair array that needs to be passed to the
 * SDK.
 *
 * This is used by the Discoverer to allocate a new array. The array is needed
 * to be passed to the SDK.
 *
 * \param size The size needed for the array.
 *
 * \return A pointer to the array, or NULL if not enough memory.
 */
typedef VNCDiscoverySDKKeyValPair *VNCCALL
VNCDiscovererAllocKeyValPairArray(size_t size);

/**
 * \brief Notifies that the Discoverer is scanning a potential entity.
 *
 * Informs the SDK that an entity has been detected and that the Discoverer is
 * scanning it to establish it is a valid entity. It is expected that once the
 * scanning is complete the Discoverer will call either of
 * VNCDiscovererNotifyEntityAppeared, or VNCDiscovererNotifyEntityDisappeared.
 *
 * This method should only be called when the Discoverer is running (has been
 * started successfully).
 *
 * It is not necessary to call thie method before calling
 * VNCDiscovererNotifyEntityAppeared.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer Pointer to the Discoverer that is scanning the entity.
 * \param pEntity Pointer to the entity that is being scanned. This entity will
 *                  not be presented to the client application before receiving
 *                  VNCDiscovererNotifyEntityAppeared.
 * \param pEventHandle The Entity can have its own Event Handle that will be
 *                  monitored for the lifetime of the Entity, while the
 *                  Discoverer is started. If the Handle is set to NULL then 
 *                  the Handle won't be monitored by the SDK. If this is
 *                  not NULL and is valid, then the Discoverer needs to
 *                  implement VNCDiscovererHandleEntityEvent. The handle can
 *                  only be changed while the Discoverer is stopped, or during
 *                  a call to VNCDiscovererHandleEntityEvent, or
 *                  VNCDiscovererHandleEntityEventTargetTime. If the Entity
 *                  doesn't wish to have the handle monitored temporarily it
 *                  should set it to VNCDiscovererNoEventHandle. The same 
 *                  handle be used in the VNCDiscovererNotifyEntityAppeared 
 *                  call.
 *
 * \see VNCDiscovererNotifyEntityAppeared, VNCDiscovererNotifyEntityDisappeared
 */
typedef void VNCCALL
VNCDiscovererNotifyScanningEntity(void *pContext, 
                          VNCDiscoverer *pDiscoverer,
                          VNCDiscovererEntity *pEntity);

/**
 * \brief Tells the Discoverer whether a certain severity will get logged.
 *
 * This method is used by the Discoverer to find out if a certain severity will
 * be logged to the client application. If the severity does not match the
 * client application options, then no logging would be performed. This method
 * can be used by the Discoverer to avoid composing unnescessary logs.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param severity The severity of the potential log.
 *
 * \return 1 if the severity would get logged, 0 otherwise.
 */
typedef vnc_int32_t VNCCALL
VNCDiscovererIsSeverityLogged(void *pContext, 
                          VNCDiscovererLogSeverity severity);

/**
 * \brief Asks the SDK to set a timer for a handle of the Discoverer.
 *
 * This call asks the SDK to call back the Discoverer, for a specific handle,
 * after a certain time period. The handle must be known to the SDK (i.e. must
 * be present in the handles returned by VNCDiscovererGetMultipleEventHandlesEx).
 *
 * The SDK will only call back once, so for further notifications this methods
 * should be called again.
 *
 * The method should only be called when the Discoverer is started, otherwise
 * it will have no effect. Make sure the SDK is notified that the Discoverer is
 * started before making this call.
 *
 * To stop an existing timer, set the timeout value to VNCDiscoverySDKNoTimeout.
 *
 * To get an instant callback, set the timeout value to VNCDisocverySDKInstantCall.
 *
 * This method must not be used if the Discoverer is not using multiple
 * handles. The Discoverer must call VNCDiscovererSetTimer in that
 * case, because calling this function will have no effect.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer The Discoverer for which to set the timer.
 * \param pEventHandle The handle for which to set the timer. Should be non-NULL
 * and the value should be different from VNCDiscovererNoEventHandle. The
 * timeout will be set for the event handle with the given value.
 * \param timeout The timeout value. It should not be negative. If you want to
 *            stop the timer set to VNCDiscoverySDKNoTimeout.
 *
 * \see VNCDiscovererHandleTimeoutForEventHandle
 * \see VNCDiscovererGetMultipleEventHandlesEx
 * \see VNCDiscovererGetEventHandle, VNCDiscovererSetTimer
 */
typedef void VNCCALL
VNCDiscovererSetTimerForHandle(void *pContext,
                                VNCDiscoverer *pDiscoverer,
                                VNCDiscovererEventHandle *pEventHandle,
                                VNCDiscoverySDKTimeoutMicroseconds timeout);

/**
 * \brief Notifies the SDK of updates to the event handle list.
 *
 * Notifies the Discovery SDK that the event handle list has changed. This
 * should be called to ensure that the SDK picks up the modified event handle
 * list when the discoverer is using multiple event handles.
 *
 * It is not necessary to call this from within the SDK thread because the
 * event handle list will be re-checked by the SDK before waiting for an event
 * again.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDiscoverer The Discoverer for which the event list has been updated.
 *
 * \see VNCDiscovererGetMultipleEventHandlesEx
 */
typedef void VNCCALL
VNCDiscovererEventHandlesChanged(void *pContext, VNCDiscoverer *pDiscoverer);

/**
 * \brief Stores Discoverer data.
 *
 * This call allows the discoverer to store some data based on a string key.
 * This means that other discoverers can share data with multiple instances of
 * itself, or with other discoverers, by name.
 *
 * The data is reference counted by the SDK. The first call to set data will
 * make the reference count to be 1.
 *
 * To ensure that race conditions are avoided, the Discovery SDK will
 * not overwrite the existing data if the data is present. Instead it will
 * increase the reference count of the existing data and return a pointer to
 * the existing data.
 *
 * To unset any existing data call VNCDiscovererUnrefData. Any set that doesn't
 * return NULL must be matched by an unref.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDataKey The name of the key where to store the data. The same name
 * can be used by multiple discoverers to point to the same data. Should not be
 * NULL. It is recommended to use a Java-style, backwards DNS-style scheme. For
 * example some example data could be stored under
 * "com.realvnc.discoverer.example".
 * \param pData The data to store. Does not replace any existing data. Must not
 * be NULL.
 *
 * \return If there is no previous data set, then this returns pData if the
 * setting was successful.  If data has been previously set, this returns
 * a pointer to the existing data (and increase the reference count to it).
 * If out of memory, or NULL was passed in, it will return NULL.
 *
 * \see VNCDiscovererRefData
 * \see VNCDiscovererUnrefData
 */
typedef void* VNCCALL
VNCDiscovererSetData(void* pContext, const char* pDataKey, void* pData);

/**
 * \brief Retrieves stored Discoverer data and increases the reference count.
 *
 * This retrieves the stored data, by key, if any. Any ref that doesn't return
 * NULL must be matched by an unref.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDataKey The name of the key where data is stored. Must not be NULL.
 *
 * \return Pointer to the stored data if present, or NULL if not present, or
 * not enough memory. If a pointer is returned, the reference count has been
 * increased and VNCDiscovererUnrefData must be called when the data is no
 * longer needed.
 *
 * \see VNCDiscovererSetData
 * \see VNCDiscovererUnrefData
 */
typedef void* VNCCALL
VNCDiscovererRefData(void* pContext, const char* pDataKey);

/**
 * \brief Decreases the reference count to stored Discoverer data.
 *
 * If the reference count reaches 0, then the SDK will remove any reference to
 * the data, and it will not be possible to retrieve the data anymore.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDataKey The name of the key where data is stored. Must not be NULL.
 *
 * \return The new reference count. If 0, then the SDK does not reference the
 * data anymore. If -1, then the data was not stored by the SDK.
 *
 * \see VNCDiscovererSetData
 * \see VNCDiscovererRefData
 */
typedef vnc_int32_t VNCCALL
VNCDiscovererUnrefData(void* pContext, const char* pDataKey);

/**
 * \brief Subscribes the Discoverer to a Device Provider.
 *
 * This call subscribes the Discoverer to a Device Provider and ensures the
 * Device Provider is available for the Discoverer. The Discovery SDK will try
 * to ensure that the Discoverer Provider is started while the subscribed
 * Discoverer is started. Once the Discoverer confirms that it started, without
 * an error occurring, all the Device Providers that it is subscribed to will be
 * started.
 *
 * Whenever the Device Providers stops, for any reason, the Discoverer is
 * notified of this through the VNCDiscovererDeviceProviderStopped call.
 *
 * If a Discoverer subscribes to a Device Provider while it's started, the
 * SDK will attempt to start the Device Provider straight away.
 *
 * The recommended place to call this is from VNCDiscovererInit, but the
 * Discoverer is free to call this at any other point afterwards.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pName The name of the Device Provider that the discoverer wishes to
 * subscribe to.
 * \param ppDeviceProvider Location of a pointer to a device provider. On
 * return, if successful, the pointer will be set to the device provider
 * instance.
 *
 * \return VNCDiscoverySDKErrorNone if subscribing was successful,
 * VNCDiscoverySDKErrorOutOfMemory if there is not enough memory to create the
 * subscription, VNCDiscoverySDKErrorLibraryNotLoaded if the Device Provider
 * can not be loaded.
 *
 * \see VNCDiscovererUnsubscribeFromDeviceProvider,
 * VNCDiscovererDeviceProviderStarted, VNCDiscovererDeviceProviderStopped
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererSubscribeToDeviceProvider(void* pContext, const char* pName,
    VNCDiscovererDeviceProvider** ppDeviceProvider);

/**
 * \brief Unsubscribes the Discoverer from a Device Provider.
 *
 * This call unsubscribes the Discoverer from the Device Provider. The
 * Discoverer would stop receiving any notifications from the SDK that are
 * related with the Device Provider.
 *
 * All devices held by the Discoverer, which were found through the Device
 * Provider, are released automatically.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDeviceProvider The Device Provider that the discoverer wishes to
 * unsubscribe from.
 *
 * \see VNCDiscovererSubscribeToDeviceProvider,
 * VNCDiscovererDeviceProviderStarted, VNCDiscovererDeviceProviderStopped
 */
typedef void VNCCALL
VNCDiscovererUnsubscribeFromDeviceProvider(void* pContext, 
    VNCDiscovererDeviceProvider* pDeviceProvider);

/**
 * \brief Sets a property for the Device Provider with the given name.
 *
 * The Discoverer must be subscribed to the Device Provider.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDeviceProvider The Device Provider.
 * \param pProperty The property to be set.
 * \param pValue The value to be set to.
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully set,
 *         VNCDiscoverySDKErrorNotSupported if the Device Provider doesn't support
 *         this property, or another error specific for the Device Provider.
 *
 * \see VNCDiscovererSubscribeToDeviceProvider,
 * VNCDiscovererGetDeviceProviderProperty
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererSetDeviceProviderProperty(void* pContext, 
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const char* pProperty,
    const char* pValue);

/**
 * \brief Gets a property of the Device Provider with the given name.
 *
 * The Discoverer must be subscribed to the Device Provider.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDeviceProviderName The Device Provider.
 * \param pProperty The property to be set.
 * \param ppValue The value that will hold the result.
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully retrieved,
 *         VNCDiscoverySDKErrorNotSupported if the Device Provider doesn't
 *         support this property, or another error specific for the Device
 *         Provider.
 *
 * \see VNCDiscovererSubscribeToDeviceProvider,
 * VNCDiscovererSetDeviceProviderProperty
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererGetDeviceProviderProperty(void* pContext, 
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const char* pProperty,
    char** ppValue);

/**
 * \brief Releases a device that has been offered to the Discoverer.
 *
 * This call should be issued by the Discoverer when no longer interested in
 * a device that has been offered to it. The SDK might offer the device to
 * another Discoverer.
 *
 * The Discoverer will no longer be notified of the removal of a device once
 * this call is made.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pDeviceProvider The Device Provider.
 * \param pDevice The device that the Discoverer is releasing.
 *
 * \see VNCDiscovererUnsubscribeFromDeviceProvider, VNCDiscovererOfferDevice,
 * VNCDiscovererRemoveDevice
 */
typedef void VNCCALL
VNCDiscovererReleaseDevice(void* pContext, 
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const VNCDeviceInfo* pDevice);

/**
 * \brief performs a local feature check.
 *
 * This call returns true if any of the feature IDs supplied are provided by a
 * license previously added locally to the Discovery SDK instance using
 * addLicense().
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param featureIDs
 * \param featureIDCount
 * \param pResult On successful return, pResult will be set to true if at least
 * one of the feature IDs is provided by a local license, and false otherwise.
 *
 * \return VNCDiscoverySDKErrorNone if the feature was checked successfully, or
 *         VNCDiscoverySDKErrorInvalidParameter if featureIDs or pResult are
 *         NULL, or if featureIDCount is 0.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererLocalFeatureCheck(void* pContext, const vnc_uint32_t* featureIDs,
    size_t featureIDCount, vnc_uint8_t* pResult);

/**
 * \brief Retrieves the utility functions supported by the SDK.
 *
 * This should be called only once the context has been set by the SDK (in
 * VNCDiscovererSetContext).
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param pUtils A pointer to a structure which will be filled with the utility
 * function.
 * \param utilsSize The size of the structure.
 *
 * \see vncdiscoverysdkutils.h
 * \see VNCDiscoverySDKUtils
 */
typedef void VNCCALL
VNCDiscovererGetUtils(void *pContext,
    VNCDiscoverySDKUtils *pUtils, size_t utilsSize);

/**
 * Returns a string explanation of the error code.
 *
 * \param error The error code to convert to string.
 *
 * \return The string representation of the error code. If the error code is
 * not known this information will be contained in the returned string.
 */
typedef const char *VNCCALL
VNCDiscovererErrorToString(VNCDiscoverySDKError error);

/**
 * \brief Retrieves the Discoverer's dynamic context.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * 
 * \return A pointer to the Discoverer's dynamic context.
 *
 * \see vncdiscoverysdkutils.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscovererDynamicContext VNCCALL
VNCDiscovererGetDynamicContext(void *pContext);

/**
 * \brief Notification that the discoverer has started a new thread which will be
 * used for SDK calls.
 *
 * This notification is mainly required on Android where threads used in the
 * SDK must be registered with the JVM. It is important to that if the
 * discoverer thread is already registered with the JVM, for example if it
 * was originally created in Java, this notification must *not* be called.
 *
 * \param pContext The context of the SDK for the Discoverer.
 */
typedef void VNCCALL
VNCDiscovererNotifyThreadStarted(void *pContext);

/**
 * \brief Notification that a thread previously registered with notifyThreadStarted
 * is about to be terminated.
 *
 * \param pContext The context of the SDK for the Discoverer.
 * \param error The final outcome error of the thread.
 */
typedef void VNCCALL
VNCDiscovererNotifyThreadEnded(void *pContext, VNCDiscoverySDKError error);

/**
 * \brief The Interface of the Support API.
 *
 * The Discoverer should use this interface to notify the client Application,
 * or to log, or allocate/free memory.
 *
 * These calls must not be invoked until discoverer construction has completed.
 * In particular, the discoverer must not take any action which would result in
 * VNCDiscovererNotifyThreadStarted being invoked in the constructor.
 * VNCDiscovererStart will usually be the correct place to start threads.
 */
typedef struct
{
  /** Major component of the SDK's version number. */
  vnc_int32_t versionMajor;
  /** Minor component of the SDK's version number. */
  vnc_int32_t versionMinor;
  /** Patch component of the SDK's version number. */
  vnc_int32_t versionPatch;
  /** Build number component of the SDK's version number. */
  vnc_int32_t versionBuild;

  /**
   * Notification once the Start operation is complete.
   */
  VNCDiscovererNotifyStarted *notifyStarted;
  /**
   * Notification once the Stop operation is complete.
   */
  VNCDiscovererNotifyStopped *notifyStopped;
  /**
   * Asks the SDK to set a timer for the Discoverer.
   */
  VNCDiscovererSetTimer *setTimer;
  /**
   * Notification that an entity has been discovered.
   */
  VNCDiscovererNotifyEntityAppeared *notifyEntityAppeared;
  /**
   * Asks the SDK to set a timer for an Entity.
   */
  VNCDiscovererSetEntityTimer *setEntityTimer;
  /**
   * Notification that an existing entity has changed.
   */
  VNCDiscovererNotifyEntityChanged *notifyEntityChanged;
  /**
   * Notification that an existing entity has disappeared.
   */
  VNCDiscovererNotifyEntityDisappeared *notifyEntityDisappeared;
  /**
   * Returns the entity value requested.
   */
  VNCDiscovererReturnEntityValue *returnEntityValue;
  /**
   * Notifies of the outcome of the post operation.
   */
  VNCDiscovererNotifyEntityValuePosted *notifyEntityValuePosted;
  /* Address book callbacks */
  /**
   * Notifies of the outcome of the publish entity operation.
   */
  VNCDiscovererNotifyEntityPublished *notifyEntityPublished;
  /**
   * Notifies of the outcome of the unpublish entity operation.
   */
  VNCDiscovererNotifyEntityUnpublished *notifyEntityUnpublished;
  /* Logging */
  /**
   * Logs a string for the Discoverer.
   */
  VNCDiscovererLog *log;
  /* Memory alloc support */
  /**
   * Allocates the memory for a string for the Discoverer.
   */
  VNCDiscovererAllocString *allocString;
  /**
   * Frees the memory of a string allocated by the SDK.
   */
  VNCDiscovererFreeString *freeString;
  /**
   * Allocates a new key-value pair array that needs to be passed to the SDK.
   */
  VNCDiscovererAllocKeyValPairArray *allocKeyValPairArray;
  /**
   * Notifies that the Discoverer is scanning a potential entity.
   */
  VNCDiscovererNotifyScanningEntity *notifyScanningEntity;
  /**
   * Tells the Discoverer whether a certain severity will get logged.
   */
  VNCDiscovererIsSeverityLogged *isSeverityLogged;
  /**
   * Asks the SDK to set a timer for a handle of the Discoverer.
   */
  VNCDiscovererSetTimerForHandle *setTimerForHandle;
  /**
   * Notifies the SDK of updates to the event handle list.
   */
  VNCDiscovererEventHandlesChanged *eventHandlesChanged;
  /**
   * Stores Discoverer data.
   */
  VNCDiscovererSetData *setData;
  /**
   * Retrieves stored Discoverer data and increases the reference count.
   */
  VNCDiscovererRefData *refData;
  /**
   * Decreases the reference count to stored Discoverer data.
   */
  VNCDiscovererUnrefData *unrefData;

  /**
   * Subscribes the Discoverer to a Device Provider.
   */
  VNCDiscovererSubscribeToDeviceProvider *subscribeToDeviceProvider;
  /**
   * Unsubscribes the Discoverer from a Device Provider.
   */
  VNCDiscovererUnsubscribeFromDeviceProvider *unsubscribeFromDeviceProvider;
  /**
   * Sets a property for the Device Provider with the given name.
   */
  VNCDiscovererSetDeviceProviderProperty *setDeviceProviderProperty;
  /**
   * Gets a property of the Device Provider with the given name.
   */
  VNCDiscovererGetDeviceProviderProperty *getDeviceProviderProperty;
  /**
   * Releases a device that has been offered to the Discoverer.
   */
  VNCDiscovererReleaseDevice *releaseDevice;
  /**
   * Performs a local feature check.
   */
  VNCDiscovererLocalFeatureCheck *localFeatureCheck;
  /**
   * Retrieves the utility functions supported by the SDK.
   */
  VNCDiscovererGetUtils *getUtils;
  /**
   * Returns a string explanation of the error code.
   */
  VNCDiscovererErrorToString *errorToString;
  /**
   * Gets the Discoverer's dynamic context.
   */
  VNCDiscovererGetDynamicContext* getDynamicContext;
  /**
   * Notification that the discoverer has started a new thread which will be
   * used for SDK calls.
   */
  VNCDiscovererNotifyThreadStarted* notifyThreadStarted;
  /**
   * Notification that a thread previously registered with notifyThreadStarted
   * is about to be terminated.
   */
  VNCDiscovererNotifyThreadEnded* notifyThreadEnded;
  /**
   * Frees a previously allocated offer parameter object.
   */
  VNCDeviceProviderFreeOfferParameter *freeOfferParameter;
} VNCDiscovererSupportAPI;

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDISCOVERERSUPPORTAPI_H__) */

